<?php

/**
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2022 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace NRFramework\Helpers\Controls;

defined('_JEXEC') or die;

class Control
{
    /**
     * Finds the value and unit in a string.
     * 
     * @param   string  $value
     * 
     * @return  array
     */
	public static function findUnitinValue($value = '')
	{
		if (!$value || !is_scalar($value))
		{
			return;
		}

		$pattern = '/^([\d.]+)(\D+)$/';
		if (preg_match($pattern, $value, $matches, PREG_OFFSET_CAPTURE) === 1)
		{
			$numericValue = $matches[1][0];
			$unit = $matches[2][0];

			return [
				'value' => $numericValue,
				'unit' => $unit
			];
		}

		return null;
	}

    /**
     * Parses the given value to a CSS value.
     * 
     * @param   mixed   $value
     * @param   string  $unit
     * 
     * @return  string
     */
    public static function getParsedValue($value = '', $unit = '')
    {
        if ($value === null || $value === false || $value === '')
        {
            return;
        }

        // Scalar
        if (is_scalar($value))
        {
            $val = $value;
            if ($value_data = self::findUnitinValue($val))
            {
                $val = $value_data['value'];
                $unit = $value_data['unit'];
            }
            return $val . ($val > 0 ? $unit : '');
        }
        else if (is_array($value))
        {
			// If all values are empty, abort
			$isEmptyArray = array_filter($value, fn($str) => $str === null || $str === false || $str === '' || (is_array($str) && empty($str)));
			if (count($isEmptyArray))
			{
				return;
			}

			// Apply spacing positions
			if ($positions = self::findSpacingPositions($value))
			{
				$return = [];
				foreach ($positions as $pos)
				{
					$return[$pos] = isset($value[$pos]) && $value[$pos] !== '' ? $value[$pos] : 0;
				}
				if (empty($return))
				{
					return;
				}
				$value = $return;
			}

            /**
             * All values are duplicates, return only 1 number with their unit.
             * 
             * Example: Given [5, 5, 5, 5] to print the margin in pixels, do not return `margin: 5px 5px 5px 5px`.
             * Rather return `margin: 5px`
             */
            if (count($value) === 4 && count(array_unique($value)) === 1)
            {
				$value = reset($value);
                $suffix = $unit;
                if ($value_data = self::findUnitinValue($value))
                {
                    $value = $value_data['value'];
                    $suffix = $value_data['unit'];
                }

				if (is_array($value))
				{
					return;
				}
				
                return $value . ($value > 0 ? $suffix : '');
            }

            /**
             * If we were given 4 values and first/third & second/forth values are the same then return these only.
             * 
             * Example: Given[5, 10, 5, 10] to print the margin in pixels, do not return `margin: 5px 10px 5px 10px`.
             * Rather return `margin: 5px 10px`
             */
            $keys = array_keys($value);
            if (count($value) === 4 && $value[$keys[0]] === $value[$keys[2]] && $value[$keys[1]] === $value[$keys[3]])
            {
                $value1 = $value[$keys[0]];
                $suffix1 = $suffix2 = $unit;
                $value2 = $value[$keys[1]];
                
                if ($value_1 = self::findUnitinValue($value1))
                {
                    $value1 = $value_1['value'];
                    $suffix1 = $value_1['unit'];
                }
                if ($value_2 = self::findUnitinValue($value2))
                {
                    $value2 = $value_2['value'];
                    $suffix2 = $value_2['unit'];
                }
                
                return $value1 . ($value1 > 0 ? $suffix1 : '') . ' ' . $value2 . ($value2 > 0 ? $suffix2 : '');
            }

            // Different values
            $data = [];
            foreach ($value as $key => $_value)
            {
                $val = $_value;
                $suffix = $unit;
                if ($value_data = self::findUnitinValue($val))
                {
                    $val = $value_data['value'];
                    $suffix = $value_data['unit'];
                }
                $data[] = $val . ($val > 0 ? $suffix : '');
            }

            return implode(' ', $data);
        }

        return;
    }

    /**
     * Finds an array of positions of the given value that
     * relates to margin/padding or border radius.
     * 
     * @param   array  $value
     * 
     * @return  array
     */
    public static function findSpacingPositions($value = [])
    {
		if (!is_array($value) || !count($value))
		{
			return;
		}
		
        $margin_padding = [
            'top',
            'right',
            'bottom',
            'left'
        ];

        $border_radius = [
            'top_left',
            'top_right',
            'bottom_right',
            'bottom_left'
        ];

		$keys = array_keys($value);

		// Is margin/padding
		if (in_array($keys[0], $margin_padding))
		{
			return $margin_padding;
		}
		// Is border radius
		else if (in_array($keys[0], $border_radius))
		{
			return $border_radius;
		}

		return;
    }
}