<?php
/**
 * Project:                 sh404SEF - SEO extension for Joomla!
 *
 * @author           Yannick Gaultier
 * @copyright        (c) Yannick Gaultier - Weeblr llc - 2021
 * @package          sh404SEF
 * @license          http://www.gnu.org/copyleft/gpl.html GNU/GPL
 * @version          4.23.2.4269
 * @date                2021-06-09
 */

namespace Weeblr\Wblib\V_SH4_4269\System;

use Weeblr\Wblib\V_SH4_4269\Wb;
use Weeblr\Wblib\V_SH4_4269\Joomla\StringHelper\StringHelper;

// no direct access
defined('_JEXEC') || defined('WBLIB_V_SH4_4269_ROOT_PATH') || die;

/**
 * Provide a few string manipulation methods
 *
 * @since    0.2.1
 *
 */
class Strings
{

	const NONE = 'none';
	const LOWERCASE = 'lowercase';
	const UPPERCASE = 'uppercase';
	const UCFIRST = 'ucfirst';

	/**
	 * Performs a preg_replace, wrapping it to catch errors
	 * caused by bad characters or otherwise
	 *
	 * @param   string  $pattern  RegExp pattern
	 * @param   string  $replace  RegExp replacement
	 * @param   string  $subject  RegExp subject
	 * @param   string  $ref      Optional reference, to be logged in case of error
	 *
	 * @return    string    the result of preg_replace operation
	 */
	public static function pr($pattern, $replace, $subject, $ref = '')
	{
		static $pageUrl = null;

		$tmp = preg_replace($pattern, $replace, $subject);
		if (is_null($tmp))
		{
			$pageUrl = is_null($pageUrl) ? (empty($_SERVER['REQUEST_URI']) ? '' : ' on page ' . $_SERVER['REQUEST_URI']) : $pageUrl;
			Log::libraryError(
				sprintf('%s::%d: %s', __METHOD__, __LINE__,
					'RegExp failed: error code: ' . preg_last_error() . ', ' . $pageUrl . (empty($ref) ? '' : ' (' . $ref . ')')
				)
			);

			return $subject;
		}
		else
		{
			return $tmp;
		}
	}

	/**
	 * Format into K and M for large number
	 * 0 -> 9999 : literral
	 * 10 000 -> 999999 : 10K -> 999,9K (max one decimal)
	 * > 1000 000 : 1M -> 1,9M (max 1 decimals)
	 *
	 * @param $n
	 *
	 * @return string
	 */
	public static function formatIntForTitle($n)
	{

		if ($n < 10000)
		{
			return (int) $n;
		}
		else if ($n < 1000000)
		{
			$n = $n / 100.0;
			$n = floor($n) / 10;
			$n = sprintf('%.1f', $n) . 'K';
		}
		else
		{
			$n = $n / 100000;
			$n = floor($n) / 10;
			$n = sprintf('%.1f', $n) . 'M';
		}

		return $n;
	}

	/**
	 * Explode a string about a delimiter, then store each part
	 * into an array, after trimming characters at both ends.
	 * Only non-empty cleaned items are added to the returned array.
	 *
	 * @param   string  $string
	 * @param   string  $delimiter
	 * @param   string  $caseHandling  none | lowercase | uppercase | ufcirst
	 *
	 * @return array
	 */
	public static function stringToCleanedArray($string, $delimiter = ',', $caseHandling = self::NONE)
	{
		$output = array();
		$bits   = explode($delimiter, $string);
		if (!empty($bits))
		{
			foreach ($bits as $bit)
			{
				$cleaned = StringHelper::trim($bit);
				if (!empty($cleaned))
				{
					switch ($caseHandling)
					{
						case self::LOWERCASE:
							$output[] = StringHelper::strtolower($cleaned);
							break;
						case self::UPPERCASE:
							$output[] = StringHelper::strtoupper($cleaned);
							break;
						case self::UCFIRST:
							$output[] = StringHelper::ucfirst($cleaned);
							break;
						default:
							$output[] = $cleaned;
							break;
					}
				}
			}
		}

		return $output;
	}

	/**
	 * @package     Joomla.Platform
	 * @subpackage  Utilities
	 *
	 * @copyright   Copyright (C) 2005 - 2016 Open Source Matters, Inc. All rights reserved.
	 * @license     GNU General Public License version 2 or later; see LICENSE
	 */

	/**
	 * Method to extract key/value pairs out of a string with XML style attributes
	 *
	 * @param   string  $string  String containing XML style attributes
	 *
	 * @return  array  Key/Value pairs for the attributes
	 *
	 * @since   11.1
	 */
	public static function parseAttributes($string)
	{

		$attr     = array();
		$retarray = array();

		// Let's grab all the key/value pairs using a regular expression
		preg_match_all('/([\w:-]+)[\s]?=[\s]?"([^"]*)"/i', $string, $attr);

		if (is_array($attr))
		{
			$numPairs = count($attr[1]);

			for ($i = 0; $i < $numPairs; $i++)
			{
				$retarray[$attr[1][$i]] = $attr[2][$i];
			}
		}

		return $retarray;
	}

	public static function wrapJsonValue($value, $type = 's')
	{
		return '{{' . $type . '}}' . $value . '{{/' . $type . '}}';
	}

	/**
	 * Return passed string as a pretty-printed JSON string, if
	 * PHP version allows.
	 *
	 * @param   string  $json
	 *
	 * @return false|string
	 */
	public static function jsonPrettyPrint($json, $humanReadable = false)
	{
		static $displayOptions = null;
		if (is_null($displayOptions))
		{
			// display options based on PHP version
			$displayOptions = false;
			$displayOptions = defined('JSON_NUMERIC_CHECK') ? $displayOptions | JSON_NUMERIC_CHECK : $displayOptions;
			$displayOptions = defined('JSON_UNESCAPED_SLASHES') ? $displayOptions | JSON_UNESCAPED_SLASHES : $displayOptions;
			$displayOptions = defined('JSON_UNESCAPED_UNICODE') ? $displayOptions | JSON_UNESCAPED_UNICODE : $displayOptions;
		}

		$encoded = json_encode($json, $humanReadable ? $displayOptions | JSON_PRETTY_PRINT : $displayOptions);

		return preg_replace(
			'~{{s}}(.*){{/s}}~uUs',
			'$1',
			$encoded
		);
	}

	/**
	 * Return passed string as a pretty-printed JSON string, if
	 * PHP version allows. Also UNESCAPE all slashes, as slashes
	 * escaping will always happen on PHP less than 5.4
	 * While valid json, escaping slashes can sometimes cause issues
	 * and is a waste anyway.
	 *
	 * Meant to unescape URLs in json fields. Do not use if regular content
	 * may have escaped slashes that should stay escaped.
	 *
	 * @param   string  $json
	 *
	 * @return mixed|string|void
	 */
	public static function jsonPrettyPrintAndUnescapeSlashes($json)
	{
		static $shouldUnescapeSlashes = null;

		if (is_null($shouldUnescapeSlashes))
		{
			$shouldUnescapeSlashes = !defined('JSON_UNESCAPED_SLASHES');
		}

		$encodedJson = self::jsonPrettyPrint($json);
		if ($shouldUnescapeSlashes)
		{
			$encodedJson = str_replace('\\/', '/', $encodedJson);
		}

		return $encodedJson;
	}

	/**
	 * Make a string safe to use as an HTML id attribute
	 *
	 * @param   string  $id
	 *
	 * @return string
	 */
	public static function asHtmlId($id)
	{
		$id = str_replace(
			array('[', ']'),
			'',
			htmlspecialchars(
				$id,
				ENT_QUOTES,
				'UTF-8'
			)
		);
		$id = str_replace(array('/', '\\', '.'), '_', $id);

		return $id;
	}

	/**
	 * Extract the content of an HTML tag from a string. Typically, body, head, html.
	 * Alway return the first instance.
	 *
	 * @param   string  $buffer
	 * @param   string  $tag
	 *
	 * @return mixed
	 */
	public static function getTagInBuffer($buffer, $tag)
	{
		if (!$buffer || !$tag)
		{
			return $buffer;
		}

		$pattern = '~<' . $tag . '[^>]*>.*</' . $tag . '[^>]*>~iUsu';

		$matched = preg_match(
			$pattern,
			$buffer,
			$matches
		);

		return $matched
			? $matches[0]
			: '';
	}

	// utility function to insert data into an html buffer, after, instead or before
	// one or more instances of a tag. If last parameter is 'first', then only the
	// first occurence of the tag is replaced, or the new value is inserted only
	// after or before the first occurence of the tag
	public static function tagInBuffer($buffer, $tag, $value, $options = [])
	{
		if (!$buffer || !$tag)
		{
			return $buffer;
		}
		$bits = explode($tag, $buffer);
		if (count($bits) < 2)
		{
			return $buffer;
		}

		$firstOnly = Wb\arrayGet($options, 'firstOnly', false);
		$where     = Wb\arrayGet($options, 'where', 'instead');

		$result   = $bits[0];
		$maxCount = count($bits) - 1;
		switch ($where)
		{
			case 'instead':
				for ($i = 0; $i < $maxCount; $i++)
				{
					$result .= ($firstOnly == 'first' ? ($i == 0 ? $value : $tag) : $value) . $bits[$i + 1];
				}
				break;
			case 'after':
				for ($i = 0; $i < $maxCount; $i++)
				{
					$result .= $tag . ($firstOnly == 'first' ? ($i == 0 ? $value : $tag) : $value) . $bits[$i + 1];
				}
				break;
			default:
				for ($i = 0; $i < $maxCount; $i++)
				{
					$result .= ($firstOnly == 'first' ? ($i == 0 ? $value : $tag) : $value) . $tag . $bits[$i + 1];
				}
				break;
		}

		return $result;
	}

	public static function pregTagInBuffer($buffer, $tag, $value, $options = [])
	{
		if (!$buffer || !$tag)
		{
			return $buffer;
		}

		$isRegExp        = Wb\arrayGet($options, 'isRegExp', false);
		$wholeWordsOnly  = Wb\arrayGet($options, 'wholeWordsOnly', false);
		$firstOnly       = Wb\arrayGet($options, 'firstOnly', false);
		$where           = Wb\arrayGet($options, 'where', 'instead');
		$isCaseSensitive = Wb\arrayGet($options, 'isCaseSensitive', false);

		if ($isRegExp)
		{
			$pattern = $tag;
		}
		else
		{
			$protectedPattern = preg_quote(
				$tag,
				'~'
			);
			$subPattern       = $wholeWordsOnly
				? '(?<!\pL)' . $protectedPattern . '(?!\pL)'
				: '(' . $protectedPattern . ')';
			$pattern          = '~' . $subPattern . '~'
				. ($isCaseSensitive ? '' : 'i')
				. 'Usu';
		}

		switch ($where)
		{
			case 'instead':
				$replacement = $value;
				break;
			case 'after':
				$replacement = '$1' . $value;
				break;
			default:
				$replacement = $value . '$1';
				break;
		}

		$result = preg_replace($pattern, $replacement, $buffer, $firstOnly ? 1 : -1);
		if (empty($result))
		{
			$result = $buffer;
			Log::libraryError(
				'wbLib', '%s::%s::%d: %s', __CLASS__, __METHOD__, __LINE__,
				'RegExp failed: invalid character in regular expression.'
			);
		}

		return $result;
	}
}
