<?php
/**
 * Project:                 sh404SEF - SEO extension for Joomla!
 *
 * @package                 sh404SEF
 * @copyright               (c) Yannick Gaultier - Weeblr llc - 2021
 * @author                  Yannick Gaultier
 * @license                 http://www.gnu.org/copyleft/gpl.html GNU/GPL
 * @version                 4.23.2.4269
 *
 * 2021-06-09
 */

namespace Weeblr\Wblib\V_SH4_4269\Seo;

use Weeblr\Wblib\V_SH4_4269\Wb;
use Weeblr\Wblib\V_SH4_4269\Base;
use Weeblr\Wblib\V_SH4_4269\System;

// Security check to ensure this file is being included by a parent file.
defined('_JEXEC') || defined('WBLIB_V_SH4_4269_ROOT_PATH') || die;

/**
 * A class to detect search engines requests.
 *
 */
class Searchengine extends Base\Base
{
	const NONE = '';
	const BING = 'bing';
	const GOOGLE = 'google';

	/**
	 * @var string Internal id of this search engine.
	 */
	public $id = '';

	/**
	 * @var string Public name of this search engine.
	 */
	public $name = '';

	/**
	 * @var bool Whether to validate search engines requesting IP address against domains.
	 */
	protected $validateIp = true;

	/**
	 * @var string[] List of possible user agents.
	 */
	protected $userAgents = [];

	/**
	 * @var string[] List of possible Ip addresses.
	 */
	protected $allowedIps = [];

	/**
	 * @var string[] List of possible hosts, each in the form .google.com (ie with a leading dot).
	 */
	protected $allowedHosts = [];

	/**
	 * @var string[][] A list of known IP addresses to use for development.
	 */
	protected $devIps = [
		self::BING   => '157.55.39.84',
		self::GOOGLE => '66.249.66.1'
	];

	/**
	 * Stores options.
	 *
	 * @param   array  $options  Can inject custom factory and platform.
	 */
	public function __construct($options = [])
	{
		parent::__construct($options);

		$this->validateIp = Wb\arrayGet(
			$options,
			'validateIp',
			true
		);
	}

	/**
	 * Whether this search engine is making the current request.
	 *
	 * @param   string  $userAgent
	 * @param   string  $ip
	 *
	 * @return bool
	 */
	public function isRequesting($userAgent, $ip)
	{
		if ('dev' == WBLIB_V_SH4_4269_OP_MODE)
		{
			$ip = Wb\arrayGet(
				$this->devIps,
				$this->id,
				$ip
			);
		}

		return $this->isUserAgent($userAgent)
			&& $this->isIpAddress($ip)
			&& $this->isAllowedHost($ip);
	}

	/**
	 * Whether user agent matches that of this search engine.
	 *
	 * @param   string  $userAgent
	 *
	 * @return bool
	 */
	protected function isUserAgent($userAgent)
	{
		return Wb\contains(
			$userAgent,
			$this->userAgents
		);
	}

	/**
	 * Whether requesting IP address matches that of this search engine.
	 *
	 * @param   string  $ip
	 *
	 * @return bool
	 */
	protected function isIpAddress($ip)
	{
		if (empty($this->allowedIps))
		{
			return true;
		}

		return in_array(
			$ip,
			$this->allowedIps
		);
	}

	/**
	 * Whether requesting IP address is from a host from this search engine.
	 *
	 * @param   string  $ip
	 *
	 * @return bool
	 */
	protected function isAllowedHost($ip)
	{
		if (!$this->validateIp)
		{
			return true;
		}

		if (empty($this->allowedHosts))
		{
			return true;
		}

		$host = System\Http::getReverseDns($ip);
		if (false === $host)
		{
			return false;
		}

		if (Wb\Endswith(
			'.' . $host,
			$this->allowedHosts
		))
		{
			$hostIp = System\Http::getForwardDns($host);

			return $ip == $hostIp;
		}

		return false;
	}
}
